'use client';

import React, { useState, useEffect } from 'react';
import { useParams } from 'next/navigation';
import { 
  User, 
  Hotel, 
  Calendar, 
  MapPin, 
  CreditCard, 
  Users,
  Bed,
  Utensils,
  Star
} from 'lucide-react';
import DetailLayout from '@/components/ui/DetailLayout';
import { renderCustomer, renderStatusBadge, accommodationStatusConfig } from '@/components/ui/TableRenderers';
import type { AccommodationRequest } from '@/types';
import PermissionGuard from '@/components/guards/PermissionGuard';

interface Comment {
  id: string;
  author: string;
  content: string;
  timestamp: Date;
  type: 'comment' | 'system' | 'status_change';
  isInternal?: boolean;
}

export default function AccommodationDetailPage() {
  return (
    <PermissionGuard requiredPermissions={['accommodation_view']}>
      <AccommodationDetailPageContent />
    </PermissionGuard>
  );
}

function AccommodationDetailPageContent() {
  const params = useParams();
  const requestId = params.id as string;
  
  const [request, setRequest] = useState<AccommodationRequest | null>(null);
  const [loading, setLoading] = useState(true);
  const [comments, setComments] = useState<Comment[]>([]);

  useEffect(() => {
    fetchRequestDetails();
    fetchComments();
  }, [requestId]);

  const fetchRequestDetails = async () => {
    try {
      setLoading(true);
      
      // Mock data
      const mockRequest: AccommodationRequest = {
        id: requestId,
        customerId: 'c1',
        customer: {
          id: 'c1',
          type: 'individual',
          firstName: 'Ahmet',
          lastName: 'Yılmaz',
          phone: '+905551234567',
          email: 'ahmet@email.com',
          isVip: false,
          isActive: true,
          createdAt: new Date(),
          updatedAt: new Date()
        },
        destination: 'Antalya',
        checkInDate: new Date('2025-01-15'),
        checkOutDate: new Date('2025-01-20'),
        nights: 5,
        guests: [
          {
            id: 'g1',
            firstName: 'Ahmet',
            lastName: 'Yılmaz',
            isChild: false
          },
          {
            id: 'g2',
            firstName: 'Ayşe',
            lastName: 'Yılmaz',
            isChild: false
          }
        ],
        totalGuests: 2,
        roomType: 'double',
        roomCount: 1,
        budgetMin: 5000,
        budgetMax: 7000,
        currency: 'TRY',
        specialRequests: 'Deniz manzaralı oda tercihi',
        mealPlan: 'breakfast',
        hotelCategory: 'standard',
        status: 'pending',
        createdAt: new Date('2025-01-10'),
        updatedAt: new Date('2025-01-10')
      };
      
      setRequest(mockRequest);
    } catch (error) {
      console.error('Error fetching request details:', error);
    } finally {
      setLoading(false);
    }
  };

  const fetchComments = async () => {
    const mockComments: Comment[] = [
      {
        id: '1',
        author: 'Sistem',
        content: 'Konaklama talebi oluşturuldu',
        timestamp: new Date('2025-01-10T10:00:00'),
        type: 'system'
      }
    ];
    setComments(mockComments);
  };

  const handleAddComment = (content: string, isInternal: boolean) => {
    const newComment: Comment = {
      id: Date.now().toString(),
      author: 'Agent Mehmet',
      content,
      timestamp: new Date(),
      type: 'comment',
      isInternal
    };
    setComments(prev => [...prev, newComment]);
  };

  const handleStatusChange = (newStatus: string) => {
    if (!request) return;
    setRequest(prev => prev ? { ...prev, status: newStatus as AccommodationRequest['status'], updatedAt: new Date() } : null);
  };

  const statusOptions = [
    { value: 'pending', label: 'Beklemede' },
    { value: 'searching', label: 'Aranıyor' },
    { value: 'quoted', label: 'Fiyat Verildi' },
    { value: 'confirmed', label: 'Onaylandı' },
    { value: 'booked', label: 'Rezerve Edildi' },
    { value: 'cancelled', label: 'İptal Edildi' },
    { value: 'completed', label: 'Tamamlandı' }
  ];

  if (loading || !request) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="animate-pulse space-y-4">
          <div className="h-8 bg-gray-200 rounded w-64"></div>
          <div className="h-64 bg-gray-200 rounded w-96"></div>
        </div>
      </div>
    );
  }

  return (
    <DetailLayout
      title={`Konaklama Talebi #${request.id}`}
      subtitle={`${request.destination} - ${request.nights} gece`}
      backUrl="/accommodation/requests"
      recordId={request.id}
      recordType="accommodation"
      comments={comments}
      onAddComment={handleAddComment}
      onStatusChange={handleStatusChange}
      statusOptions={statusOptions}
      currentStatus={request.status}
      currentUser={{ id: 'agent1', name: 'Acente', type: 'agent' }}
    >
      <div className="space-y-6">
        {/* Status Card */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <div className="flex items-center justify-between mb-4">
            <h3 className="text-lg font-semibold text-gray-900">Talep Durumu</h3>
            {renderStatusBadge(request.status, accommodationStatusConfig)}
          </div>
        </div>

        {/* Customer Info */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
            <User className="h-5 w-5 mr-2" />
            Müşteri Bilgileri
          </h3>
          {renderCustomer(request.customer)}
        </div>

        {/* Accommodation Details */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
            <Hotel className="h-5 w-5 mr-2" />
            Konaklama Detayları
          </h3>
          <div className="grid grid-cols-2 gap-6">
            <div>
              <div className="flex items-center space-x-2 mb-2">
                <MapPin className="h-4 w-4 text-gray-400" />
                <span className="text-sm text-gray-500">Destinasyon</span>
              </div>
              <p className="text-lg font-semibold">{request.destination}</p>
            </div>
            <div>
              <div className="flex items-center space-x-2 mb-2">
                <Calendar className="h-4 w-4 text-gray-400" />
                <span className="text-sm text-gray-500">Tarihler</span>
              </div>
              <p className="font-medium">
                {request.checkInDate.toLocaleDateString('tr-TR')} - {request.checkOutDate.toLocaleDateString('tr-TR')}
              </p>
              <p className="text-sm text-gray-500">{request.nights} gece</p>
            </div>
          </div>
        </div>

        {/* Room & Guest Info */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
            <Bed className="h-5 w-5 mr-2" />
            Oda ve Misafir Bilgileri
          </h3>
          <div className="grid grid-cols-3 gap-4">
            <div>
              <span className="text-gray-500">Misafir Sayısı:</span>
              <p className="font-medium flex items-center">
                <Users className="h-4 w-4 mr-1" />
                {request.totalGuests} kişi
              </p>
            </div>
            <div>
              <span className="text-gray-500">Oda Sayısı:</span>
              <p className="font-medium">{request.roomCount} oda</p>
            </div>
            <div>
              <span className="text-gray-500">Oda Tipi:</span>
              <p className="font-medium">{request.roomType}</p>
            </div>
          </div>
          
          {request.mealPlan && (
            <div className="mt-4 flex items-center">
              <Utensils className="h-4 w-4 text-gray-400 mr-2" />
              <span className="text-gray-500 mr-2">Yemek Planı:</span>
              <span className="font-medium">{request.mealPlan}</span>
            </div>
          )}
        </div>

        {/* Budget */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
            <CreditCard className="h-5 w-5 mr-2" />
            Bütçe
          </h3>
          <div className="flex items-center justify-between">
            <span className="text-gray-600">Bütçe Aralığı:</span>
            <span className="text-lg font-bold text-green-600">
              {request.budgetMin?.toLocaleString()} - {request.budgetMax?.toLocaleString()} {request.currency}
            </span>
          </div>
        </div>

        {/* Special Requests */}
        {request.specialRequests && (
          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
            <h3 className="text-lg font-semibold text-gray-900 mb-4">Özel İstekler</h3>
            <p className="text-gray-700">{request.specialRequests}</p>
          </div>
        )}

        {/* Hotel Info (if booked) */}
        {request.hotelName && (
          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
            <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
              <Star className="h-5 w-5 mr-2" />
              Otel Bilgileri
            </h3>
            <div className="space-y-2">
              <p className="font-medium text-lg">{request.hotelName}</p>
              {request.confirmationNumber && (
                <p className="text-sm text-blue-600 font-mono">
                  Rezervasyon No: {request.confirmationNumber}
                </p>
              )}
              {request.totalPrice && (
                <p className="text-lg font-bold text-green-600">
                  {request.totalPrice.toLocaleString()} {request.currency}
                </p>
              )}
            </div>
          </div>
        )}
      </div>
    </DetailLayout>
  );
}